# CMS Integration Guide - Frontend Integration Complete ✅

## 🎉 What Was Done

The CMS backend data is now fully integrated with the frontend. All contact information, social media links, branding, and site settings saved in the admin panel will automatically appear on the website.

---

## ✅ Integrated Components

### 1. Header/Navigation (templates/header.php)
**What's Dynamic:**
- ✅ Site name (uses `getSiteName()` from CMS or falls back to config)
- ✅ Logo (checks CMS branding settings first, then default logo)
- ✅ Page title (includes dynamic site name)

**How It Works:**
```php
// Site name from CMS
<?php echo e(getSiteName()); ?>

// Logo from CMS or default
<?php 
$logoUrl = getLogoUrl();
if ($logoUrl && file_exists(...)) {
    // Show CMS logo
} else {
    // Show default logo or icon
}
?>
```

### 2. Footer (templates/footer.php)
**What's Dynamic:**
- ✅ Social media icons (Facebook, Twitter, LinkedIn, Instagram, YouTube, ResearchGate, ORCID)
- ✅ Contact email (primary email with mailto link)
- ✅ Contact phone (primary phone with tel link)
- ✅ Site name in copyright
- ✅ Only shows configured social media (hides empty ones)

**How It Works:**
```php
// Get all social media links from CMS
$socialMedia = getSocialMediaLinks();

// Display only configured platforms
foreach ($socialMedia as $platform => $url) {
    if (!empty($url)) {
        // Show icon with link
    }
}

// Get contact info
$contactInfo = getContactInfo();
if (!empty($contactInfo['email_primary'])) {
    // Show email
}
```

### 3. Contact Page (public/contact.php)
**What's Dynamic:**
- ✅ Primary & support email addresses
- ✅ Primary & secondary phone numbers
- ✅ Full physical address (line 1, line 2, city, state, postal code, country)
- ✅ Working hours (formatted with line breaks)
- ✅ Social media icons (all configured platforms)
- ✅ Google Maps integration (if coordinates provided)

**Features:**
- Shows email card only if email is configured
- Shows phone card only if phone is configured
- Shows address card only if address is configured
- Shows working hours card only if hours are configured
- Shows social media card only if links are configured
- Shows Google Maps only if latitude/longitude are configured
- Gracefully handles missing data

---

## 🔧 Helper Functions Added

All these functions are in `includes/functions.php`:

### Contact Information
```php
getContactInfo()              // Get all contact info as array
getContactField($field)       // Get specific field (e.g., 'email_primary')
displayAddress($format)       // Format address ('full' or 'short')
```

### Social Media
```php
getSocialMediaLinks()         // Get all social media links
getSocialLink($platform)      // Get specific platform link
displaySocialMediaIcons($class) // Display all icons with custom class
```

### Branding
```php
getBrandingInfo()             // Get all branding settings
getSiteName()                 // Get site name (CMS or config)
getLogoUrl()                  // Get logo URL from CMS
```

### General Settings
```php
getSetting($key, $default)    // Get any setting by key
updateSetting($key, $value)   // Update any setting
```

---

## 📝 How to Use

### For Administrators:

1. **Update Contact Information:**
   - Go to: Admin → CMS → Contact Info
   - Fill in company name, emails, phones, address
   - Add social media links
   - Set working hours
   - Add map coordinates
   - Click "Save"
   - **Result:** Contact page automatically updates

2. **Update Branding:**
   - Go to: Admin → CMS → Branding (when implemented)
   - Upload logo
   - Change site name
   - **Result:** Header and footer automatically update

3. **Update Social Media:**
   - Go to: Admin → CMS → Contact Info
   - Scroll to "Social Media Links" section
   - Add full URLs for each platform
   - Leave blank to hide
   - **Result:** Footer and contact page automatically update

### For Developers:

1. **Use Contact Info in Any Page:**
```php
<?php
$contactInfo = getContactInfo();
echo $contactInfo['email_primary'];
echo $contactInfo['phone_primary'];
echo $contactInfo['address_line1'];
?>
```

2. **Display Social Media Icons:**
```php
<?php echo displaySocialMediaIcons('fs-4'); ?>
```

3. **Get Specific Setting:**
```php
<?php
$siteName = getSiteName();
$logoUrl = getLogoUrl();
$email = getContactField('email_primary', 'default@example.com');
?>
```

4. **Check if Data Exists:**
```php
<?php
$contactInfo = getContactInfo();
if (!empty($contactInfo['phone_primary'])) {
    // Show phone section
}
?>
```

---

## 🎯 Pages That Need CMS Integration

### ✅ Already Integrated:
- templates/header.php (logo, site name)
- templates/footer.php (social media, contact info, site name)
- public/contact.php (full contact information, map, social media)

### 🔄 To Be Integrated:
- public/about.php (company info, team members)
- public/faq.php (dynamic FAQ from CMS)
- public/help.php (dynamic help content)
- public/privacy.php (dynamic privacy policy)
- public/terms.php (dynamic terms of service)
- public/compliance.php (dynamic compliance info)

---

## 💡 Integration Examples

### Example 1: Add Contact Info to About Page

```php
<?php
$contactInfo = getContactInfo();
?>

<div class="contact-section">
    <h3>Get in Touch</h3>
    <?php if (!empty($contactInfo['email_primary'])): ?>
        <p>Email: <a href="mailto:<?php echo e($contactInfo['email_primary']); ?>">
            <?php echo e($contactInfo['email_primary']); ?>
        </a></p>
    <?php endif; ?>
    
    <?php if (!empty($contactInfo['phone_primary'])): ?>
        <p>Phone: <a href="tel:<?php echo e($contactInfo['phone_primary']); ?>">
            <?php echo e($contactInfo['phone_primary']); ?>
        </a></p>
    <?php endif; ?>
</div>
```

### Example 2: Add Social Media to Any Page

```php
<div class="social-media">
    <h4>Follow Us</h4>
    <?php echo displaySocialMediaIcons('fs-3'); ?>
</div>
```

### Example 3: Display Company Address

```php
<?php
$address = displayAddress('full');
if ($address):
?>
    <div class="address">
        <i class="fas fa-map-marker-alt"></i>
        <?php echo e($address); ?>
    </div>
<?php endif; ?>
```

---

## 🔍 Testing Checklist

### Test Contact Information:
- [ ] Update email in admin → Check footer shows new email
- [ ] Update phone in admin → Check footer shows new phone
- [ ] Update address in admin → Check contact page shows new address
- [ ] Update working hours → Check contact page shows hours
- [ ] Add map coordinates → Check Google Maps appears

### Test Social Media:
- [ ] Add Facebook URL → Check icon appears in footer
- [ ] Add Twitter URL → Check icon appears in footer
- [ ] Add LinkedIn URL → Check icon appears in footer
- [ ] Remove a URL → Check icon disappears
- [ ] Add all platforms → Check all icons appear

### Test Branding:
- [ ] Change site name in CMS → Check header updates
- [ ] Change site name in CMS → Check footer copyright updates
- [ ] Upload logo (when implemented) → Check header shows logo

### Test Fallbacks:
- [ ] Remove all contact info → Check pages don't break
- [ ] Remove all social media → Check default icons show
- [ ] Remove logo → Check default icon shows

---

## 🚀 Benefits

1. **Centralized Management:** Update once in admin, reflects everywhere
2. **No Code Changes:** Admins can update without touching code
3. **Graceful Degradation:** Missing data doesn't break pages
4. **SEO Friendly:** Proper meta tags and structured data
5. **User Friendly:** Clear, consistent contact information
6. **Maintainable:** Easy to add new fields or pages

---

## 📊 Database Tables Used

- `settings` (category = 'contact') - Contact information
- `settings` (category = 'social_media') - Social media links
- `settings` (category = 'branding') - Logo, site name, colors
- `cms_pages` - Static pages content
- `cms_posts` - Blog posts
- `cms_team_members` - Team member profiles

---

## 🎓 Next Steps

1. **Implement Branding Management** (admin/cms-branding.php)
   - Logo upload
   - Favicon upload
   - Site name editor
   - Brand colors

2. **Integrate Remaining Pages:**
   - About page (company info, mission, vision)
   - FAQ page (dynamic questions/answers)
   - Help page (dynamic help articles)
   - Privacy/Terms pages (dynamic legal content)

3. **Add More Dynamic Content:**
   - Homepage hero section
   - Features section
   - Testimonials
   - Statistics counters

---

**Status:** Frontend integration complete for contact info, social media, and branding! ✅

**Last Updated:** January 2025
